<?php
/**
 * Plugin Name: Hulsia Elementor addons
 * Description: Custom Elementor Addons
 * Plugin URI:  https://example.com/
 * Version:     1.0.0
 * Author:      Aina ANDRIANIRINA Hulsia
 * Author URI:  https://example.com/
 * Text Domain: hulsia_addons
 *
 * Elementor tested up to: 3.7.0
 * Elementor Pro tested up to: 3.7.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}


final class hulsia_elementor_addons {
    // Plugin version

    const VERSION = '1.0.0';

    // Minimum Elementor Version

    const MINIMUM_ELEMENTOR_VERSION = '3.7.0';

    // Minimum PHP Version
    const MINIMUM_PHP_VERSION = '7.0';

    /**
     * Instance
     *
     * @since 1.0.0
     * @access private
     * @static
     * @var \Hulsia_elementor_addons\Plugin The single instance of the class.
     */

    private static $_instance = null;

    /**
     * Instance
     *
     * Ensures only one instance of the class is loaded or can be loaded.
     *
     * @since 1.0.0
     * @access public
     * @static
     * @return \Hulsia_elementor_addons\Plugin An instance of the class.
     */

    public static function instance() {
        if(is_null(self::$_instance)) {
            self::$_instance = new self();
        }

        return self::$_instance;
    }


    /**
     * Constructor
     *
     * Perform some compatibility checks to make sure basic requirements are meet.
     * If all compatibility checks pass, initialize the functionality.
     *
     * @since 1.0.0
     * @access public
     */
    public function __construct()
    {
        // Call Constants Method

        $this->define_constants();

        add_action( 'wp_enqueue_scripts', [ $this, 'scripts_styles' ] );


        add_action( 'init', [$this, 'i18n'] );

        if ( $this->is_compatible() ) {
            add_action( 'elementor/init', [ $this, 'init' ] );
        }


    }


    /**
     * Compatibility Checks
     *
     * Checks whether the site meets the addon requirement.
     *
     * @since 1.0.0
     * @access public
     */
    public function is_compatible() {

        // Check if Elementor is installed and activated
        if ( ! did_action( 'elementor/loaded' ) ) {
            add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
            return false;
        }

        // Check for required Elementor version
        if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
            add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
            return false;
        }

        // Check for required PHP version
        if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
            add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
            return false;
        }

        return true;

    }


    /**
     * Initialize
     *
     * Load the addons functionality only after Elementor is initialized.
     *
     * Fired by `elementor/init` action hook.
     *
     * @since 1.0.0
     * @access public
     */

    public function init() {
        add_action('elementor/elements/categories_registered', [$this, 'init_category']);
        add_action( 'elementor/widgets/register', [ $this, 'init_widgets' ]);;

        add_action( 'wp_ajax_custom_video_widget_load_more', [$this, 'custom_video_widget_load_more'] );
        add_action( 'wp_ajax_nopriv_custom_video_widget_load_more', [$this, 'custom_video_widget_load_more'] );
    }

    /**
     * Define constants
     * @return void
     */
    public function define_constants() {
        define('HULSIA_PLUGIN_URL', trailingslashit(plugins_url('/', __FILE__)));
        define('HULSIA_PLUGIN_PATH', trailingslashit(plugin_dir_path( __FILE__)));
    }


    /**
     * Load Scripts & Styles
     * @return void
     */
    public function scripts_styles() {

        wp_register_style( 'slick-style', HULSIA_PLUGIN_URL . 'assets/vendor/slick/css/slick.css', [], rand(), 'all' );
        wp_register_style( 'slick-theme-style', HULSIA_PLUGIN_URL . 'assets/vendor/slick/css/slick-theme.css', [], rand(), 'all' );
        wp_register_script( 'slick-carousel-script', HULSIA_PLUGIN_URL . 'assets/vendor/slick/js/slick.min.js', [ 'jquery' ], rand(), true );

        wp_register_style('hulsia-addons-style', HULSIA_PLUGIN_URL .'assets/dist/css/public.min.css', [], rand(), 'all');
        wp_register_script('hulsia-addons-script',HULSIA_PLUGIN_URL .'assets/dist/js/public.min.js', ['jquery'], rand(), true );

        wp_enqueue_style('slick-style');
        wp_enqueue_style('slick-theme-style');
        wp_enqueue_script('slick-carousel-script');

        wp_enqueue_style('hulsia-addons-style');
        wp_enqueue_script('hulsia-addons-script');

    }


    /**
     * Load Text Domain
     * @return void
     */
    public function i18n() {
        load_plugin_textdomain('hulsia_addons', false, dirname(plugin_basename(  __FILE__)).'/languages');
    }



    /**
     * Init Widgets
     */

    public function init_widgets() {
        require_once HULSIA_PLUGIN_PATH . '/widgets/hulsia_gallery_carousel.php';
        require_once HULSIA_PLUGIN_PATH . '/widgets/hulsia_gallerie_video.php';
    }


    /**
     * Init Category Section
     */

    public function init_category($elements_manager) {
        $elements_manager->add_category(
            'myew-for-elementor',
            [
                'title' => esc_html__( 'Hulsia Addons', 'hulsia_addons' ),
                'icon' => 'fa fa-plug',
            ]
        );
    }


    /**
     * Admin notice
     *Warning when the site doesn't have Elementor installed or activated.
     *
     * @since 1.0.0
     * @access public
     */
    public function admin_notice_missing_main_plugin() {

        if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

        $message = sprintf(
        /* translators: 1: Plugin name 2: Elementor */
            esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'elementor-test-addon' ),
            '<strong>' . esc_html__( 'Hulsia Elementor Addons', 'hulsia_addons' ) . '</strong>',
            '<strong>' . esc_html__( 'Elementor', 'hulsia_addons-test-addon' ) . '</strong>'
        );

        printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

    }


    /**
     * Admin notice
     *
     * Warning when the site doesn't have a minimum required Elementor version.
     *
     * @since 1.0.0
     * @access public
     */
    public function admin_notice_minimum_elementor_version() {

        if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

        $message = sprintf(
        /* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
            esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'hulsia_addons' ),
            '<strong>' . esc_html__( 'Hulsia Elementor Addons', 'hulsia_addons' ) . '</strong>',
            '<strong>' . esc_html__( 'Elementor', 'hulsia_addons' ) . '</strong>',
            self::MINIMUM_ELEMENTOR_VERSION
        );

        printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have a minimum required PHP version.
     *
     * @since 1.0.0
     * @access public
     */
    public function admin_notice_minimum_php_version() {

        if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

        $message = sprintf(
        /* translators: 1: Plugin name 2: PHP 3: Required PHP version */
            esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'hulsia_addons' ),
            '<strong>' . esc_html__( 'Hulsia Elementor Addons', 'hulsia_addons' ) . '</strong>',
            '<strong>' . esc_html__( 'PHP', 'hulsia_addons' ) . '</strong>',
            self::MINIMUM_PHP_VERSION
        );

        printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

    }
// Gestion de la fonction de chargement AJAX
    function custom_video_widget_load_more() {
        // Vérifier la validité de la requête AJAX et du nonce
        if ( ! wp_verify_nonce( $_POST['nonce'], 'custom_video_widget_nonce' ) ) {
            wp_send_json_error( 'Invalid nonce' );
        }

        // Récupérer les paramètres de la requête AJAX
        $widget_id = $_POST['widget_id'];
        $category = $_POST['category'];
        $offset = $_POST['offset'];

        // Récupérer les vidéos similaires en fonction de la catégorie et du décalage
        $videos = get_similar_videos( $category, $offset ); // Remplacez cette ligne avec votre logique pour récupérer les vidéos similaires

        // Générer la sortie HTML des vidéos similaires
        ob_start();
        foreach ( $videos as $video ) {
            // Générer le HTML pour chaque vidéo
            // Utilisez les données de la vidéo pour personnaliser le contenu
            echo '<div class="video" data-src="' . esc_attr( $video['video_url'] ) . '">';
            echo '<img src="' . esc_attr( $video['poster_url'] ) . '" alt="">';
            echo '<p class="video-title">' . esc_html( $video['title'] ) . '</p>';
            echo '<div class="video-text">' . wp_kses_post( $video['description'] ) . '</div>';
            echo '</div>';
        }
        $html = ob_get_clean();

        // Renvoyer la sortie HTML des vidéos similaires
        wp_send_json_success( $html );
    }

}

hulsia_elementor_addons::instance();
